<?php

namespace Imanghafoori\LaravelMicroscope\Features\CheckImports;

use Illuminate\Console\Command;
use Imanghafoori\LaravelMicroscope\Analyzers\ComposerJson;
use Imanghafoori\LaravelMicroscope\ErrorReporters\ErrorPrinter;
use Imanghafoori\LaravelMicroscope\Features\CheckImports\Checks\CheckClassAtMethod;
use Imanghafoori\LaravelMicroscope\Features\CheckImports\Checks\CheckClassReferencesAreValid;
use Imanghafoori\LaravelMicroscope\Features\CheckImports\Handlers\ClassAtMethodHandler;
use Imanghafoori\LaravelMicroscope\Features\CheckImports\Handlers\PrintWrongClassRefs;
use Imanghafoori\LaravelMicroscope\Features\CheckImports\Reporters\AutoloadFiles;
use Imanghafoori\LaravelMicroscope\Features\CheckImports\Reporters\CheckImportReporter;
use Imanghafoori\LaravelMicroscope\Features\FacadeAlias\FacadeAliasesCheck;
use Imanghafoori\LaravelMicroscope\Features\FacadeAlias\FacadeAliasReplacer;
use Imanghafoori\LaravelMicroscope\Features\FacadeAlias\FacadeAliasReporter;
use Imanghafoori\LaravelMicroscope\Features\Thanks;
use Imanghafoori\LaravelMicroscope\FileReaders\FilePath;
use Imanghafoori\LaravelMicroscope\ForPsr4LoadedClasses;
use Imanghafoori\LaravelMicroscope\Foundations\PhpFileDescriptor;
use Imanghafoori\LaravelMicroscope\Iterators\BladeFiles;
use Imanghafoori\LaravelMicroscope\Iterators\ChecksOnPsr4Classes;
use Imanghafoori\LaravelMicroscope\Iterators\ClassMapIterator;
use Imanghafoori\LaravelMicroscope\Iterators\FileIterators;
use Imanghafoori\LaravelMicroscope\LaravelPaths\LaravelPaths;
use Imanghafoori\LaravelMicroscope\SearchReplace\CachedFiles;
use Imanghafoori\LaravelMicroscope\SpyClasses\RoutePaths;
use Imanghafoori\LaravelMicroscope\Traits\LogsErrors;
use Imanghafoori\TokenAnalyzer\ImportsAnalyzer;
use Imanghafoori\TokenAnalyzer\ParseUseStatement;
use JetBrains\PhpStorm\Pure;

class CheckImportsCommand extends Command
{
    use LogsErrors;

    protected $signature = 'check:imports
        {--force : fixes without asking}
        {--w|wrong : Only reports wrong imports}
        {--e|extra : Only reports extra imports}
        {--f|file= : Pattern for file names to scan}
        {--d|folder= : Pattern for file names to scan}
        {--s|nofix : avoids the automatic fixes}
    ';

    protected $description = 'Checks the validity of use statements';

    protected $customMsg = '';

    /**
     * @var array<int, class-string<\Imanghafoori\LaravelMicroscope\Iterators\Check>>
     */
    private $checks = [
        1 => CheckClassAtMethod::class,
        2 => CheckClassReferencesAreValid::class,
        3 => FacadeAliasesCheck::class,
    ];

    public function handle()
    {
        event('microscope.start.command');
        $this->line('');
        $this->info('Checking imports and class references...');

        FacadeAliasesCheck::$command = $this->getOutput();

        if ($this->option('nofix')) {
            ClassAtMethodHandler::$fix = false;
            FacadeAliasesCheck::$handler = FacadeAliasReporter::class;
            CheckClassReferencesAreValid::$wrongClassRefsHandler = PrintWrongClassRefs::class;
        }

        if (file_exists($path = CachedFiles::getPathForPattern().'check_imports.php')) {
            CheckClassReferencesAreValid::$cache = (require $path) ?: [];
        }

        if ($this->option('force')) {
            FacadeAliasReplacer::$forceReplace = true;
        }

        if ($this->option('wrong')) {
            CheckClassReferencesAreValid::$checkExtra = false;
            unset($this->checks[3]); // avoid checking facades
        }

        if ($this->option('extra')) {
            CheckClassReferencesAreValid::$checkWrong = false;
            unset($this->checks[3]); // avoid checking facades
        }

        $fileName = ltrim($this->option('file'), '=');
        $folder = ltrim($this->option('folder'), '=');
        $folder = rtrim($folder, '/\\');

        $routeFiles = FilePath::removeExtraPaths(RoutePaths::get(), $folder, $fileName);

        $autoloadedFilesGen = FilePath::removeExtraPaths(
            ComposerJson::autoloadedFilesList(base_path()),
            $folder,
            $fileName
        );

        $paramProvider = self::getParamProvider();

        $checks = $this->checks;
        unset($checks[1]);

        $classMapStats = ClassMapIterator::iterate(base_path(), $checks, $paramProvider, $fileName, $folder);

        $routeFiles = FileIterators::checkFiles($routeFiles, $checks, $paramProvider);
        $autoloadedFilesGen = FileIterators::checkFilePaths($autoloadedFilesGen, $checks, $paramProvider);

        $foldersStats = FileIterators::checkFolders(
            $checks,
            self::getLaravelFolders(),
            $paramProvider,
            $fileName,
            $folder
        );

        $psr4Stats = ForPsr4LoadedClasses::check($this->checks, $paramProvider, $fileName, $folder);

        $checks = $this->checks;
        unset($checks[3]); // avoid checking facades aliases in blade files.
        $bladeStats = BladeFiles::check($checks, $paramProvider, $fileName, $folder);

        $errorPrinter = ErrorPrinter::singleton($this->output);

        /**
         * @var string[] $messages
         */
        $messages = [];
        $messages[0] = CheckImportReporter::totalImportsMsg();
        $messages[1] = Reporters\Psr4Report::printAutoload($psr4Stats, $classMapStats);
        $messages[2] = CheckImportReporter::header();
        $messages[3] = self::getFilesStats();
        $messages[4] = Reporters\BladeReport::getBladeStats($bladeStats);
        $messages[5] = Reporters\LaravelFoldersReport::foldersStats($foldersStats);
        $messages[6] = CheckImportReporter::getRouteStats($routeFiles);
        $messages[7] = AutoloadFiles::getLines($autoloadedFilesGen);
        $errorPrinter->flushErrors();
        $messages[8] = Reporters\SummeryReport::summery($errorPrinter->errorsCounts);

        if (! ImportsAnalyzer::$checkedRefCount) {
            $messages = ['<options=bold;fg=yellow>No imports were found!</> with filter: <fg=red>"'.($fileName ?: $folder).'"</>'];
        }

        $this->finishCommand($errorPrinter);
        $this->getOutput()->writeln(implode(PHP_EOL, array_filter($messages)));

        $errorPrinter->printTime();

        if (Thanks::shouldShow()) {
            self::printThanks($this);
        }

        if ($cache = CheckClassReferencesAreValid::$cache) {
            self::writeCacheContent($cache);
            self::printThanks($this);
        }

        $this->line('');

        return ErrorCounter::getTotalErrors() > 0 ? 1 : 0;
    }

    #[Pure]
    private static function printThanks($command)
    {
        $command->line(PHP_EOL);
        foreach (Thanks::messages() as $msg) {
            $command->line($msg);
        }
    }

    #[Pure]
    private static function getParamProvider()
    {
        return function (PhpFileDescriptor $file) {
            $imports = ParseUseStatement::parseUseStatements($file->getTokens());

            return $imports[0] ?: [$imports[1]];
        };
    }

    #[Pure]
    private static function getFilesStats(): string
    {
        $filesCount = ChecksOnPsr4Classes::$checkedFilesCount;

        return $filesCount ? CheckImportReporter::getFilesStats($filesCount) : '';
    }

    /**
     * @return array<string, \Generator>
     */
    #[Pure]
    private static function getLaravelFolders()
    {
        return [
            'config' => LaravelPaths::configDirs(),
            'migrations' => LaravelPaths::migrationDirs(),
        ];
    }

    private static function writeCacheContent(array $cache): void
    {
        $folder = CachedFiles::getPathForPattern();
        ! is_dir($folder) && mkdir($folder);
        $content = CachedFiles::getCacheFileContents($cache);
        $path = $folder.'check_imports.php';
        file_exists($path) && chmod($path, 0777);
        file_put_contents($path, $content);
    }
}
